/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package renderdata;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.util.Properties;

import com.inet.report.*;
import com.inet.viewer.RenderData;
import com.inet.viewer.ReportViewer;

/**
 * This class implements a RenderData with a given Engine. The Engine will be executed later in the current Java VM. You
 * can load the engine from a template or you can create it with RDC. The difference to EngineRenderData of the public
 * API is that you can use an existing Engine object instead of a report template name. That this will not work: <li>If
 * it is a serialized Engine from a server. The Engine must execute locally. <li>If you need to set data with
 * Engine.setData(x).
 */
public class EngineRenderDataFromExistingEngine extends EngineRenderData {

    private final Engine referenceEngine;

    /**
     * Create a new RenderData with an existing Engine
     * @param engine a loaded or created Engine which was not execute
     * @param props prompts and other parameters
     * @throws ReportException if Engine is not initialized or finished.
     */
    public EngineRenderDataFromExistingEngine( Engine engine, Properties props ) throws ReportException {
        this( props, engine );
        engine.getFields(); //throws ReportException if Engine is not initialized or finished.
    }

    /**
     * Create a new RenderData with an existing Engine without checking the state of the engine.
     * @param engine a loaded or created Engine which was not execute
     * @param props prompts and other parameters
     */
    private EngineRenderDataFromExistingEngine( Properties props, Engine engine ) {
        super( "" );
        getProperties().putAll( props );
        this.referenceEngine = engine;
    }

    /**
     * Creates or load an <code>Engine</code> and set all needed properties. Do NOT execute the <code>Engine</code>
     * here!
     * @param props Properties to use for creating the engine.
     * @return the created engine
     * @throws ReportException if the specified export format is invalid.
     */
    @Override
    protected Engine createEngine( Properties props ) throws ReportException {
        try {
            String exportFormat = props.getProperty( "export_fmt" );

            // Create a copy the referenceEngine with the target format
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            RDC.saveEngine( baos, referenceEngine );
            baos.flush();
            InputStream in = new ByteArrayInputStream( baos.toByteArray() );
            Engine eng = RDC.loadEngine( referenceEngine.getReportFile(), in, exportFormat, null );

            // Set the parameters, if you want set data or other parameters then you need to do it here
            setEngineParams( eng, props, true );

            return eng;
        } catch( ReportException ex ) {
            ex.printStackTrace();
            throw ex;
        } catch( Throwable e ) {
            e.printStackTrace();
            ReportException repEx = new ReportException( e.getMessage(), 0 );
            repEx.initCause( e );
            throw repEx;
        }
    }

    /**
     * "Clones" this RenderData object with all its properties and returns the copy. Useful for deriving from existing
     * RenderData objects by copying them and adding or changing properties. This method is called by the viewer for
     * drilling down, for example - the drilldown property is set on the copy while all other properties remain the
     * same, and the copy is used to open a new report view.
     * @return A cloned copy of this RenderData object with all its properties.
     * @see ReportViewer#addNewReportView(RenderData)
     * @see RenderData#getProperties()
     */
    @Override
    public RenderData getCopy() {
        return new EngineRenderDataFromExistingEngine( getProperties(), referenceEngine );
    }
}
